/**
 * Copyright (C) 2015 by Liangliang Nan (liangliang.nan@gmail.com)
 * https://3d.bk.tudelft.nl/liangliang/
 *
 * This file is part of Easy3D. If it is useful in your research/work,
 * I would be grateful if you show your appreciation by citing it:
 * ------------------------------------------------------------------
 *      Liangliang Nan.
 *      Easy3D: a lightweight, easy-to-use, and efficient C++
 *      library for processing and rendering 3D data. 2018.
 * ------------------------------------------------------------------
 * Easy3D is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 3
 * as published by the Free Software Foundation.
 *
 * Easy3D is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <easy3d/fileio/resources.h>
#include <easy3d/util/file_system.h>


namespace easy3d {

    namespace resource {

        // resource directory (containing color maps, shaders, textures, etc.)
        std::string directory() {
            // first check if the resources has been defined in the
            std::string dir = EASY3D_RESOURCES_DIR;
            if (file_system::is_directory(dir)) {
                LOG_FIRST_N(1, INFO) << "resources directory: " << dir;
                return dir;
            }

            std::string parent = file_system::executable_directory();
            dir = parent + "/resources";
            if (file_system::is_directory(dir)) {
                LOG_FIRST_N(1, INFO) << "resources directory: " << dir;
                return dir;
            }
            else {
                // For macOS, if reached here, we may need to move "up" three times, because
                // macOS puts the executable file in an application bundle, e.g.,
                // "PolyFit.app/Contents/MacOS/PolyFit". Some IDEs may also put the 'ext' in
                // Debug/Release subfolder, so we may to try four times up at most.
                parent = file_system::parent_directory(parent);
                dir = parent + "/resources";
                if (file_system::is_directory(dir)) {
                    LOG_FIRST_N(1, INFO) << "resources directory: " << dir;
                    return dir;
                }
                else {
                    for (int i = 0; i < 4; ++i) {
                        parent = file_system::parent_directory(parent);
                        dir = parent + "/resources";
                        if (file_system::is_directory(dir)) {
                            LOG_FIRST_N(1, INFO) << "resources directory: " << dir;
                            return dir;
                        }
                    }
                }
                // if still could not find it, show an error and return the current working directory
                LOG_FIRST_N(1, ERROR) << "could not find the resources directory";
                return file_system::current_working_directory();
            }
        }


#ifdef USE_BUNNY

        // const int bunny_num_vertices = 453;
        // const int bunny_num_faces = 902;

        std::vector<vec3> bunny_vertices =
        {
            vec3(-0.3343919814f, -0.06225900725f, 0.1330070049f),
            vec3(-0.3501889706f, 0.1477690041f, 0.1503539979f),
            vec3(-0.2342009842f, 0.1743069887f, 0.3438110054f),
               vec3(-0.2002589852f, -0.09374901652f, 0.2852070034f),
                vec3(0.003519999795f, 0.1593649834f, 0.4752080142f),
                vec3(0.00185599993f, -0.09858201444f, 0.4192030132f),
                vec3(-0.2528019845f, -0.237537995f, 0.0936659947f),
                vec3(-0.1629009843f, -0.2069050223f, 0.2379839867f),
                vec3(0.0008649999509f, -0.2353700101f, 0.3181410134f),
                vec3(-0.4146239758f, 0.2782540023f, 0.1640830189f),
                vec3(-0.2622129619f, 0.2932459712f, 0.357333988f),
                vec3(0.004627999384f, 0.3386259675f, 0.482694f),
                vec3(-0.4021619558f, 0.4432469904f, 0.1335280091f),
                vec3(-0.2437809855f, 0.4367629886f, 0.3242750168f),
                vec3(0.005292999558f, 0.4583319724f, 0.4375920296f),
                vec3(-0.3398839831f, 0.6682109833f, -0.04114997014f),
                vec3(-0.2483819872f, 0.6274930239f, 0.2558250427f),
                vec3(0.006260999478f, 0.6315060258f, 0.3761030436f),
                vec3(-0.2162009925f, 0.8869360089f, -0.1267759502f),
                vec3(-0.1710749865f, 0.881385982f, 0.01154403854f),
                vec3(-0.1810739785f, 0.8147789836f, 0.09822303802f),
                vec3(-0.1198909953f, 0.7601529956f, 0.218786031f),
                vec3(-0.07889499515f, 0.7392809987f, 0.2767800391f),
                vec3(0.006800999399f, 0.7356609702f, 0.3109590411f),
                vec3(-0.1688419878f, 0.9203810096f, 0.1023870409f),
                vec3(-0.1040719897f, 0.9525300264f, 0.1772780418f),
                vec3(-0.1297039837f, 0.8366780281f, 0.2118480355f),
                vec3(-0.09987499565f, 0.7993810177f, 0.3109310269f),
                vec3(0.00723699946f, 0.7944390178f, 0.3616870344f),
                vec3(-0.07791299373f, 0.9216399789f, 0.2587530315f),
                vec3(0.007956999354f, 0.9316800237f, 0.2822410166f),
                vec3(-0.2522219718f, 0.5578100085f, -0.5504009724f),
                vec3(-0.2676329613f, 0.6552090049f, -0.6034190059f),
                vec3(-0.4468379617f, 0.4661589861f, -0.1185169742f),
                vec3(-0.4594879746f, 0.3113409877f, -0.09301698208f),
                vec3(-0.3706449568f, 0.159454003f, -0.1001079902f),
                vec3(-0.3719839752f, 0.011044004f, -0.09199099988f),
                vec3(-0.3289449811f, -0.08865899593f, -0.0982690081f),
                vec3(-0.2824519575f, -0.3115009964f, -0.01886201277f),
                vec3(-0.3524029851f, -0.1449020058f, -0.1313409954f),
                vec3(-0.364125967f, -0.2023879886f, -0.2002990097f),
                vec3(-0.2839649618f, -0.02366799116f, -0.2318689972f),
                vec3(-0.2989429832f, -0.3697159886f, -0.1552180201f),
                vec3(-0.2937869728f, -0.4190970063f, -0.1218560115f),
                vec3(-0.2901629806f, -0.1078239828f, -0.2907969952f),
                vec3(-0.2641649842f, -0.03634699062f, -0.2728489935f),
                vec3(-0.2285669893f, -0.2903089821f, -0.3725729883f),
                vec3(-0.1904309839f, -0.4219169915f, -0.2869970202f),
                vec3(-0.1910389811f, -0.5071179867f, -0.2409730107f),
                vec3(-0.2872719765f, 0.06544401497f, -0.2764309943f),
                vec3(-0.2956749797f, 0.1742000133f, -0.2808179855f),
                vec3(-0.3995369673f, 0.3761669993f, -0.3131309748f),
                vec3(-0.3926659822f, 0.4274940193f, -0.488580972f),
                vec3(-0.331668973f, 0.4660540223f, -0.5701850057f),
                vec3(-0.2822899818f, 0.5892199874f, -0.6181399822f),
                vec3(-0.3742379844f, 0.3232980371f, -0.5948820114f),
                vec3(-0.3810709715f, 0.35077703f, -0.6297230124f),
                vec3(-0.3821119666f, 0.2215770334f, -0.6240599751f),
                vec3(-0.2727009654f, -0.2591569722f, -0.5665220022f),
                vec3(-0.2567019761f, -0.2860789597f, -0.6634060144f),
                vec3(-0.2809479833f, -0.05578998104f, -0.4283590019f),
                vec3(-0.184973985f, -0.3262649775f, -0.5088940263f),
                vec3(-0.2799709737f, -0.395318985f, -0.526917994f),
                vec3(-0.2825989723f, -0.4124109745f, -0.6633930206f),
                vec3(-0.1883289814f, -0.417953968f, -0.4750930369f),
                vec3(-0.2633839846f, -0.4666039646f, -0.6633960009f),
                vec3(-0.2090629786f, -0.5093439817f, -0.6633930206f),
                vec3(-0.002043999732f, -0.5530779958f, -0.3196240366f),
                vec3(-0.001265999861f, -0.4132959843f, -0.3712600172f),
                vec3(-0.2197529823f, 0.0409210138f, -0.3397620022f),
                vec3(-0.2569859624f, 0.006349012721f, -0.2825109959f),
                vec3(-0.2717059851f, -0.001763988636f, -0.2608810067f),
                vec3(-0.09119099379f, 0.04591201991f, -0.4191839993f),
                vec3(-0.1149439961f, 0.1247390211f, -0.4297519922f),
                vec3(-0.1139699891f, 0.1885400116f, -0.3829869926f),
                vec3(-0.2430119812f, 0.2428500205f, -0.4649420083f),
                vec3(-0.3148149848f, 0.3247680366f, -0.5054020286f),
                vec3(0.002773999702f, 0.2627660334f, -0.4375259876f),
                vec3(-0.07262498885f, 0.2214400172f, -0.4177480042f),
                vec3(-0.1601119787f, 0.2934500277f, -0.4769319892f),
                vec3(0.003858999815f, 0.4439160228f, -0.4534249604f),
                vec3(-0.1203629896f, 0.4386890233f, -0.5815669894f),
                vec3(-0.09149899334f, 0.2945110202f, -0.5841910243f),
                vec3(-0.1164689958f, 0.1883080304f, -0.5998610258f),
                vec3(-0.2080319822f, 0.1346490234f, -0.5136399865f),
                vec3(-0.2357489914f, 0.04093902558f, -0.6100170016f),
                vec3(-0.3449159563f, 0.08538003266f, -0.6224870086f),
                vec3(-0.3364009559f, 0.2122980356f, -0.5318639874f),
                vec3(0.001960999798f, 0.1355470121f, -0.4595499933f),
                vec3(-0.058295995f, 0.04344001785f, -0.4305360019f),
                vec3(0.001377999899f, 0.03776201978f, -0.4495109916f),
                vec3(-0.1301349849f, -0.07914397866f, -0.5102220178f),
                vec3(0.0001419999899f, -0.1570639908f, -0.4775489867f),
                vec3(-0.1142839938f, -0.3043969572f, -0.4532060027f),
                vec3(-0.0005919999676f, -0.2854009569f, -0.4435580075f),
                vec3(-0.05621499568f, -0.3260729611f, -0.6634020209f),
                vec3(-0.0262479987f, -0.2733179629f, -0.5680099726f),
                vec3(-0.04926099628f, -0.3898539841f, -0.5310639739f),
                vec3(-0.1270959824f, -0.4793159664f, -0.6633980274f),
                vec3(-0.05838399753f, -0.3728909791f, -0.6634010077f),
                vec3(-0.303960979f, -0.625921011f, 0.05419897288f),
                vec3(-0.268593967f, -0.5027660131f, 0.1934029907f),
                vec3(-0.2771589756f, -0.4432890117f, 0.1261229813f),
                vec3(-0.2876049578f, -0.5318440199f, -0.005722023081f),
                vec3(-0.2313959897f, -0.5873870254f, -0.1212890223f),
                vec3(-0.2534749806f, -0.7565410137f, -0.0817970261f),
                vec3(-0.1951639801f, -0.7280110121f, -0.1379690319f),
                vec3(-0.1676729918f, -0.6093879938f, -0.1565730274f),
                vec3(-0.1459169835f, -0.6976000071f, -0.1690290272f),
                vec3(-0.07777599245f, -0.6225860119f, -0.214247033f),
                vec3(-0.07687298954f, -0.6963009834f, -0.2149710357f),
                vec3(-0.002340999665f, -0.6228590012f, -0.2331350297f),
                vec3(-0.002729999833f, -0.6912670135f, -0.2135260254f),
                vec3(-0.003135999665f, -0.7627310157f, -0.192628026f),
                vec3(-0.05613599718f, -0.7638059855f, -0.2012220323f),
                vec3(-0.1145889908f, -0.7707229853f, -0.1661920249f),
                vec3(-0.1551449895f, -0.7917379737f, -0.129632026f),
                vec3(-0.1836109906f, -0.8470119834f, -0.0587050356f),
                vec3(-0.1655619889f, -0.833386004f, 0.001979963621f),
                vec3(-0.2200839818f, -0.7689350247f, 0.01991396584f),
                vec3(-0.2557299733f, -0.6707820296f, 0.09030596912f),
                vec3(-0.2555939555f, -0.6633890271f, 0.1138329729f),
                vec3(-0.2263799906f, -0.6177399755f, 0.2126549631f),
                vec3(-0.003366999794f, -0.7996799946f, -0.1953420341f),
                vec3(-0.029742999f, -0.827180028f, -0.2105080336f),
                vec3(-0.003817999735f, -0.8736360073f, -0.1947830468f),
                vec3(-0.004115999676f, -0.9312679768f, -0.1579070389f),
                vec3(-0.03127999604f, -0.8894760013f, -0.1845550388f),
                vec3(-0.05988499522f, -0.8413299918f, -0.1844480336f),
                vec3(-0.1353329867f, -0.8781999946f, -0.1643320471f),
                vec3(-0.08557399362f, -0.9255470037f, -0.1709480435f),
                vec3(-0.1638329923f, -0.8971139789f, -0.09417003393f),
                vec3(-0.1384439915f, -0.9459750056f, -0.1042500436f),
                vec3(-0.08349699527f, -0.979606986f, -0.08493404835f),
                vec3(-0.004432999529f, -0.9858719707f, -0.1466420442f),
                vec3(-0.1507149786f, -0.8841109872f, 0.03264996409f),
                vec3(-0.1358919889f, -0.9454550147f, -0.0355200395f),
                vec3(-0.07061199099f, -0.9757329822f, 0.03684895858f),
                vec3(-0.004457999486f, -1.015669942f, -0.04252604395f),
                vec3(-0.004248999525f, -1.003239989f, 0.04604195431f),
                vec3(-0.08696899563f, -0.947633028f, 0.1332239509f),
                vec3(-0.003872999689f, -0.9704989791f, 0.1616049558f),
                vec3(-0.1255439818f, -0.9176779985f, 0.1400119513f),
                vec3(-0.125650987f, -0.8576020002f, 0.2502459586f),
                vec3(-0.003126999829f, -0.8788700104f, 0.2840699553f),
                vec3(-0.1591739804f, -0.8888779879f, 0.1257259548f),
                vec3(-0.1838069856f, -0.8444799781f, 0.1969699711f),
                vec3(-0.1598899812f, -0.7324799895f, 0.2917359769f),
                vec3(-0.1994949877f, -0.7798640132f, 0.2072299719f),
                vec3(-0.2061819881f, -0.6932569742f, 0.1646079719f),
                vec3(-0.1863149852f, -0.8171929717f, 0.1606889665f),
                vec3(-0.1928269863f, -0.7822710276f, 0.166705966f),
                vec3(-0.1751119792f, -0.8606209755f, 0.1100079641f),
                vec3(-0.1610219777f, -0.8551110029f, 0.05741996318f),
                vec3(-0.1723189801f, -0.8161889911f, 0.03615496308f),
                vec3(-0.1903179884f, -0.7606049776f, 0.06408297271f),
                vec3(-0.1950719804f, -0.7311040163f, 0.1291789711f),
                vec3(-0.2031259835f, -0.6805359721f, 0.410286963f),
                vec3(-0.2166769803f, -0.6422719955f, 0.3092739582f),
                vec3(-0.2415149808f, -0.587831974f, 0.3114849627f),
                vec3(-0.002208999824f, -0.7494130135f, 0.3666629791f),
                vec3(-0.0882299915f, -0.6786350012f, 0.3962649703f),
                vec3(-0.1701469868f, -0.8407840133f, 0.1095169634f),
                vec3(-0.1605209857f, -0.830649972f, 0.06776596606f),
                vec3(-0.1815459877f, -0.812146008f, 0.1398049742f),
                vec3(-0.180494979f, -0.7760869861f, 0.1485679746f),
                vec3(-0.1802549809f, -0.7441920042f, 0.1291249692f),
                vec3(-0.1862979829f, -0.7693520188f, 0.07830796391f),
                vec3(-0.1676219851f, -0.8066750169f, 0.06053896621f),
                vec3(-0.1898759902f, -0.8025820255f, 0.1027599648f),
                vec3(-0.1083399951f, -0.6571739912f, 0.4554459751f),
                vec3(-0.2415849864f, -0.6692960262f, 0.5275920033f),
                vec3(-0.265675962f, -0.6345940232f, 0.513365984f),
                vec3(-0.20307298f, -0.5815259814f, 0.4785499573f),
                vec3(-0.2667719722f, -0.6020609736f, 0.6423299909f),
                vec3(-0.2169609815f, -0.5354350209f, 0.5648459792f),
                vec3(-0.2022099793f, -0.4759440422f, 0.5254949927f),
                vec3(-0.1938879788f, -0.5206059813f, 0.4679249823f),
                vec3(-0.2658369839f, -0.5009330511f, 0.7572669983f),
                vec3(-0.24030599f, -0.4632150233f, 0.6534399986f),
                vec3(-0.3092389703f, -0.3047260344f, 0.7768679857f),
                vec3(-0.2710089684f, -0.3820180297f, 0.6830940247f),
                vec3(-0.3121109605f, -0.2866870165f, 0.6710990071f),
                vec3(-0.2687909603f, -0.3772310317f, 0.6243420243f),
                vec3(-0.302456975f, -0.3602890372f, 0.533995986f),
                vec3(-0.2636559606f, -0.4125640392f, 0.529309988f),
                vec3(-0.2823109627f, -0.4476660192f, 0.4151669741f),
                vec3(-0.2392009795f, -0.4956040382f, 0.4420959651f),
                vec3(-0.2200429887f, -0.4458770156f, 0.5690259933f),
                vec3(-0.001262999838f, -0.6020290256f, 0.3956309557f),
                vec3(-0.05734499544f, -0.572224021f, 0.4425349832f),
                vec3(-0.08892699331f, -0.5291060209f, 0.5063329935f),
                vec3(-0.12573798f, -0.6129130125f, 0.5350760221f),
                vec3(-0.1262509823f, -0.4831590354f, 0.5771700144f),
                vec3(-0.1495939791f, -0.5577309728f, 0.6115199924f),
                vec3(-0.1631879807f, -0.4910800159f, 0.6607909799f),
                vec3(-0.1724819839f, -0.4154160321f, 0.6633870006f),
                vec3(-0.1604639888f, -0.3706590235f, 0.5917099714f),
                vec3(-0.1564449817f, -0.3783020377f, 0.5363960266f),
                vec3(-0.1364959925f, -0.4252260327f, 0.4443579614f),
                vec3(-0.09556399286f, -0.4736590385f, 0.373767972f),
                vec3(-0.1041459963f, -0.4981040061f, 0.3159119785f),
                vec3(-0.0004959999351f, -0.4738170207f, 0.3841939569f),
                vec3(-0.0001829999819f, -0.4014860094f, 0.2977699637f),
                vec3(-0.1290419847f, -0.434495002f, 0.2701449692f),
                vec3(9.99999902e-05f, -0.3491379917f, 0.2729629576f),
                vec3(-0.1130599901f, -0.3855539858f, 0.2369839847f),
                vec3(0.007259999402f, 0.8833960295f, 0.01631103829f),
                vec3(0.007864998654f, 0.9561370015f, 0.1221040413f),
                vec3(-0.03284199536f, 0.9532520175f, 0.1152820438f),
                vec3(-0.08911499381f, 0.9503170252f, 0.1084490418f),
                vec3(-0.04743999615f, 0.8827559948f, 0.01472903788f),
                vec3(-0.1044579893f, 0.8820700049f, 0.01313703787f),
                vec3(-0.08643899113f, 0.6083430052f, -0.5848659873f),
                vec3(-0.1150259897f, 0.436732024f, -0.6626049876f),
                vec3(-0.0716829896f, 0.6063849926f, -0.665372014f),
                vec3(-0.257883966f, 0.6580520272f, -0.6653810143f),
                vec3(-0.27254197f, 0.5920630097f, -0.6653810143f),
                vec3(-0.3713219762f, 0.3536200225f, -0.6653820276f),
                vec3(-0.372361958f, 0.2244200259f, -0.6653810143f),
                vec3(-0.3351659775f, 0.07862302661f, -0.6653800011f),
                vec3(-0.2259989828f, 0.03898103163f, -0.6653749943f),
                vec3(-0.1067189947f, 0.1863510311f, -0.665373981f),
                vec3(-0.0817489922f, 0.2925540209f, -0.665372014f),
                vec3(0.006942999549f, 0.8583539724f, -0.09150496125f),
                vec3(0.006116999313f, 0.7699670196f, -0.2809849679f),
                vec3(0.004494999535f, 0.5597990155f, -0.5023599863f),
                vec3(-0.1986379921f, 0.8458160162f, -0.3021349609f),
                vec3(-0.2373949885f, 0.5871880054f, -0.5425440073f),
                vec3(-0.2700009644f, 0.6698610187f, -0.2794889808f),
                vec3(-0.1345469803f, 0.9590039849f, -0.1198519543f),
                vec3(-0.05208799616f, 0.9445490241f, -0.1224629581f),
                vec3(-0.124462992f, 0.8995659947f, -0.2935079634f),
                vec3(-0.04761599749f, 0.8792920113f, -0.2896429598f),
                vec3(-0.1685949862f, 0.6549310088f, -0.5291320086f),
                vec3(-0.09979299456f, 0.6458730102f, -0.5157189965f),
                vec3(-0.1861679852f, 0.7246900201f, -0.605281949f),
                vec3(-0.1129699945f, 0.7074689865f, -0.5830969214f),
                vec3(-0.1081519946f, 0.7004079819f, -0.6653749347f),
                vec3(-0.1830189824f, 0.7176300287f, -0.6653779149f),
                vec3(-0.3495289683f, 0.5119850039f, -0.334458977f),
                vec3(-0.1411819905f, 0.7981939912f, -0.4377049804f),
                vec3(-0.2126699835f, 0.7374470234f, -0.4487249553f),
                vec3(-0.2611109614f, 0.6138349771f, -0.4149449766f),
                vec3(-0.07736399025f, 0.7781130075f, -0.4314799607f),
                vec3(0.005173999351f, 0.6515920162f, -0.4252769649f),
                vec3(0.08923599124f, 0.7770929933f, -0.4317319691f),
                vec3(0.2710059583f, 0.6105769873f, -0.4157489836f),
                vec3(0.2239809781f, 0.7347739935f, -0.4493839741f),
                vec3(0.1532749832f, 0.7963910103f, -0.4381499588f),
                vec3(0.3584139645f, 0.5076490045f, -0.3355289698f),
                vec3(0.1934339851f, 0.7153249979f, -0.6659459472f),
                vec3(0.118362993f, 0.6990209818f, -0.6657169461f),
                vec3(0.123514995f, 0.7060199976f, -0.5834539533f),
                vec3(0.1968509853f, 0.7223449945f, -0.6058599353f),
                vec3(0.1097879931f, 0.6445900202f, -0.5160350204f),
                vec3(0.178655982f, 0.6528040171f, -0.529655993f),
                vec3(0.06115699559f, 0.878625989f, -0.289806962f),
                vec3(0.1382339895f, 0.8979579806f, -0.2939049602f),
                vec3(0.06693299115f, 0.9438199997f, -0.1226429567f),
                vec3(0.1495709866f, 0.9572640061f, -0.1202809587f),
                vec3(0.2809889615f, 0.6664869785f, -0.2803209722f),
                vec3(0.2465809882f, 0.5842239857f, -0.5432749987f),
                vec3(0.2117199898f, 0.8433030248f, -0.3027539849f),
                vec3(0.08696599305f, 0.2915200293f, -0.6656270027f),
                vec3(0.1106339917f, 0.1850210279f, -0.6657019854f),
                vec3(0.2280989885f, 0.03620103002f, -0.6660609841f),
                vec3(0.3377429843f, 0.0745030269f, -0.6663960218f),
                vec3(0.3767219782f, 0.2198330313f, -0.6665130258f),
                vec3(0.3772649765f, 0.3490360379f, -0.6665130258f),
                vec3(0.2814109623f, 0.5886700153f, -0.6662170291f),
                vec3(0.2675639689f, 0.6548339725f, -0.6661739945f),
                vec3(0.08074498922f, 0.6054520011f, -0.6656020284f),
                vec3(0.1220159903f, 0.435280025f, -0.6629629731f),
                vec3(0.09576699138f, 0.6072279811f, -0.5851410031f),
                vec3(0.1189439893f, 0.8807020187f, 0.01279903855f),
                vec3(0.0619439967f, 0.882085979f, 0.01456403825f),
                vec3(0.1047249958f, 0.9491299987f, 0.1081560478f),
                vec3(0.04851299524f, 0.9527530074f, 0.1151590422f),
                vec3(0.112695992f, -0.3869369924f, 0.2366429865f),
                vec3(0.1281769872f, -0.4360710084f, 0.2697569728f),
                vec3(0.1026429906f, -0.4993700087f, 0.315599978f),
                vec3(0.09453499317f, -0.4748240411f, 0.3734809756f),
                vec3(0.1362699866f, -0.4268950224f, 0.4439459741f),
                vec3(0.1570709795f, -0.3802220225f, 0.5359230042f),
                vec3(0.161349982f, -0.3726300299f, 0.5912240148f),
                vec3(0.1730349809f, -0.4175310433f, 0.6628649831f),
                vec3(0.1628079861f, -0.4930770397f, 0.6602990031f),
                vec3(0.1482499838f, -0.5595549941f, 0.6110699773f),
                vec3(0.125718981f, -0.4847020209f, 0.5767899752f),
                vec3(0.1234889925f, -0.6144400239f, 0.5346990228f),
                vec3(0.08762099594f, -0.530188024f, 0.5060660243f),
                vec3(0.05532099679f, -0.5729150176f, 0.4423649609f),
                vec3(0.2199359834f, -0.4485710263f, 0.5683609843f),
                vec3(0.238098979f, -0.4985280335f, 0.4413749576f),
                vec3(0.2817109823f, -0.4511210322f, 0.4143149555f),
                vec3(0.2638329566f, -0.4157940149f, 0.5285130143f),
                vec3(0.3032839596f, -0.3639980257f, 0.5330809951f),
                vec3(0.2696869671f, -0.3805280328f, 0.6235280037f),
                vec3(0.3142549694f, -0.2905240357f, 0.6701530218f),
                vec3(0.2720229626f, -0.3853430152f, 0.6822729707f),
                vec3(0.3114799559f, -0.3085270226f, 0.7759310007f),
                vec3(0.2402389795f, -0.4661590159f, 0.6527140141f),
                vec3(0.2656189799f, -0.5041870475f, 0.7564640045f),
                vec3(0.1925619841f, -0.5229719877f, 0.467340976f),
                vec3(0.2016049922f, -0.4784170389f, 0.5248849988f),
                vec3(0.2157429904f, -0.5380839705f, 0.5641930103f),
                vec3(0.2649689615f, -0.6053169966f, 0.6415269971f),
                vec3(0.2010309845f, -0.584002018f, 0.4779399633f),
                vec3(0.2630859613f, -0.637831986f, 0.5125669837f),
                vec3(0.2386149913f, -0.672236979f, 0.5268669724f),
                vec3(0.1053089947f, -0.6584820151f, 0.4551229775f),
                vec3(0.1839929819f, -0.8048719764f, 0.1021949649f),
                vec3(0.1615629792f, -0.8086919785f, 0.06004196778f),
                vec3(0.1807479858f, -0.771600008f, 0.07775396109f),
                vec3(0.1751679778f, -0.746367991f, 0.1285879761f),
                vec3(0.1750749797f, -0.7782639861f, 0.1480299681f),
                vec3(0.1756579876f, -0.8143330216f, 0.139264971f),
                vec3(0.1541909873f, -0.8325780034f, 0.06729096174f),
                vec3(0.1638179868f, -0.8428300023f, 0.1090129614f),
                vec3(0.08475999534f, -0.6796950102f, 0.3960039616f),
                vec3(0.2388879806f, -0.590775013f, 0.3107599616f),
                vec3(0.2133799791f, -0.6449049711f, 0.3086249828f),
                vec3(0.1996659786f, -0.6830030084f, 0.4096779823f),
                vec3(0.1901429892f, -0.7334629893f, 0.1285969764f),
                vec3(0.1848329902f, -0.7629020214f, 0.06351596862f),
                vec3(0.1660699844f, -0.8182610273f, 0.03564396128f),
                vec3(0.1543609798f, -0.8570420146f, 0.05694296211f),
                vec3(0.1685419828f, -0.8627250195f, 0.1094889641f),
                vec3(0.1873869896f, -0.7845990062f, 0.1661309749f),
                vec3(0.1804279834f, -0.8194379807f, 0.1601349711f),
                vec3(0.2018229812f, -0.6957560182f, 0.1639909744f),
                vec3(0.1942059845f, -0.7822750211f, 0.2066349685f),
                vec3(0.155437991f, -0.7344120145f, 0.2912599742f),
                vec3(0.1776959896f, -0.8466929793f, 0.1964239776f),
                vec3(0.1523049921f, -0.8907859921f, 0.1252559572f),
                vec3(0.1195459962f, -0.8591039777f, 0.2498759627f),
                vec3(0.1183689907f, -0.9191730022f, 0.1396429539f),
                vec3(0.07940999418f, -0.948652029f, 0.1329729557f),
                vec3(0.06241899729f, -0.9765470028f, 0.03664796054f),
                vec3(0.1278469861f, -0.9470700026f, -0.03591904044f),
                vec3(0.1436239779f, -0.8859130144f, 0.03220596164f),
                vec3(0.07488799095f, -0.980576992f, -0.08517304808f),
                vec3(0.1301839799f, -0.9476199746f, -0.1046560481f),
                vec3(0.1562009901f, -0.899074018f, -0.09465304017f),
                vec3(0.07736599445f, -0.9265450239f, -0.1711940467f),
                vec3(0.1277219802f, -0.8798099756f, -0.1647290438f),
                vec3(0.05266999453f, -0.8420190215f, -0.184618026f),
                vec3(0.02347699739f, -0.8898109794f, -0.1846380383f),
                vec3(0.02262599766f, -0.8274999857f, -0.2105870247f),
                vec3(0.2230889797f, -0.6204929948f, 0.2119759768f),
                vec3(0.2514439821f, -0.6664940119f, 0.1130669713f),
                vec3(0.251418978f, -0.6738870144f, 0.08953996748f),
                vec3(0.2143599838f, -0.7715950012f, 0.01925796643f),
                vec3(0.1589989811f, -0.8353739977f, 0.001489963499f),
                vec3(0.1766959876f, -0.8492180109f, -0.05924903601f),
                vec3(0.1486959904f, -0.7935990095f, -0.1300910264f),
                vec3(0.1082899943f, -0.7720879912f, -0.1665280312f),
                vec3(0.04981999472f, -0.7644540071f, -0.2013820261f),
                vec3(0.07134099305f, -0.6972090006f, -0.21519503f),
                vec3(0.07314798981f, -0.6235100031f, -0.2144750357f),
                vec3(0.1405019909f, -0.6993539929f, -0.1694610268f),
                vec3(0.1633739918f, -0.6114159822f, -0.1570730358f),
                vec3(0.1894659847f, -0.7303659916f, -0.1385500282f),
                vec3(0.2475929856f, -0.7596099973f, -0.08255402744f),
                vec3(0.227467984f, -0.5901970267f, -0.1219820231f),
                vec3(0.2847019732f, -0.5353469849f, -0.006586023141f),
                vec3(0.2757409811f, -0.4466759861f, 0.1252869815f),
                vec3(0.2666499615f, -0.5060439706f, 0.1925939918f),
                vec3(0.3000859618f, -0.6296200156f, 0.05328697339f),
                vec3(0.05544999614f, -0.3750649691f, -0.6639350057f),
                vec3(0.1228539944f, -0.4823229611f, -0.6641380191f),
                vec3(0.04651999474f, -0.3919179738f, -0.531570971f),
                vec3(0.02482399903f, -0.275105983f, -0.5684499741f),
                vec3(0.05385499448f, -0.3282239735f, -0.6639310122f),
                vec3(0.1128289923f, -0.3057879806f, -0.4535489976f),
                vec3(0.1312649846f, -0.08074598014f, -0.5106170177f),
                vec3(0.06117399782f, 0.04271001741f, -0.4307160079f),
                vec3(0.3410189748f, 0.2081500292f, -0.532886982f),
                vec3(0.3477049768f, 0.08113902807f, -0.6235330105f),
                vec3(0.2380399853f, 0.03803702444f, -0.6107320189f),
                vec3(0.211763978f, 0.1320780367f, -0.5142740011f),
                vec3(0.1206049919f, 0.1868560314f, -0.6002190113f),
                vec3(0.09698498994f, 0.2933570445f, -0.5844759941f),
                vec3(0.1276209801f, 0.4371700287f, -0.5819410086f),
                vec3(0.1659019887f, 0.2914530337f, -0.477425009f),
                vec3(0.07771999389f, 0.2205190212f, -0.4179750085f),
                vec3(0.3208919764f, 0.3208740354f, -0.5063629746f),
                vec3(0.2482139915f, 0.2398420125f, -0.4656839967f),
                vec3(0.1187639907f, 0.1871140152f, -0.3833380044f),
                vec3(0.1188159958f, 0.1233070195f, -0.430106014f),
                vec3(0.09413099289f, 0.04477701709f, -0.4194639921f),
                vec3(0.2745259702f, -0.005109988153f, -0.2617059946f),
                vec3(0.2598419785f, 0.003185012378f, -0.2832919955f),
                vec3(0.222860992f, 0.03821001574f, -0.3404310048f),
                vec3(0.2044449896f, -0.5133529902f, -0.6643800139f),
                vec3(0.2592859566f, -0.4712809622f, -0.6645470262f),
                vec3(0.185401991f, -0.4217179716f, -0.4760200381f),
                vec3(0.2791629732f, -0.4173279703f, -0.6646040082f),
                vec3(0.2771569788f, -0.4002079666f, -0.5281220078f),
                vec3(0.1830689907f, -0.3299949765f, -0.5098119974f),
                vec3(0.2825989723f, -0.05924198031f, -0.4292100072f),
                vec3(0.2548159659f, -0.290686965f, -0.6645410061f),
                vec3(0.271435976f, -0.2639659643f, -0.5677070022f),
                vec3(0.3865609765f, 0.2168700248f, -0.625221014f),
                vec3(0.3870859742f, 0.3460730314f, -0.6308829784f),
                vec3(0.3800209761f, 0.3186790347f, -0.5960209966f),
                vec3(0.2912689745f, 0.5857070088f, -0.6190069914f),
                vec3(0.3392799795f, 0.4619460404f, -0.5711979866f),
                vec3(0.4000449777f, 0.4226400256f, -0.4897779822f),
                vec3(0.4068169594f, 0.3712300062f, -0.3143489659f),
                vec3(0.3005879819f, 0.1705490202f, -0.2817179859f),
                vec3(0.2908659577f, 0.06190501153f, -0.2773039937f),
                vec3(0.1877349913f, -0.5094370246f, -0.2415450215f),
                vec3(0.1880319864f, -0.4242340028f, -0.2875690162f),
                vec3(0.227519989f, -0.2931019664f, -0.3732619882f),
                vec3(0.2665259838f, -0.03959698975f, -0.2736499906f),
                vec3(0.2915919721f, -0.1113859862f, -0.2916760147f),
                vec3(0.2919139564f, -0.4226830006f, -0.1227410138f),
                vec3(0.2975739837f, -0.373367995f, -0.1561190188f),
                vec3(0.2866029739f, -0.02716199122f, -0.2327309996f),
                vec3(0.3646629751f, -0.2068499923f, -0.2013990134f),
                vec3(0.353854984f, -0.1492280066f, -0.1324079931f),
                vec3(0.2822079659f, -0.3149600029f, -0.01971501298f),
                vec3(0.33118698f, -0.0927009955f, -0.09926600754f),
                vec3(0.3754629791f, 0.006467004307f, -0.09312000126f),
                vec3(0.3759169579f, 0.1548819989f, -0.1012359932f),
                vec3(0.466634959f, 0.3056690097f, -0.09441598505f),
                vec3(0.4558049738f, 0.4606319964f, -0.1198809743f),
                vec3(0.277464956f, 0.6518710256f, -0.6042420268f),
                vec3(0.2610219717f, 0.554666996f, -0.5511760116f),
                vec3(0.09362699091f, 0.9205890298f, 0.2584940195f),
                vec3(0.1142479926f, 0.7980700135f, 0.3106080294f),
                vec3(0.14423199f, 0.8350009918f, 0.2114340365f),
                vec3(0.119915992f, 0.9511590004f, 0.1769400388f),
                vec3(0.1840609908f, 0.9182199836f, 0.1018540338f),
                vec3(0.09243099391f, 0.7382310033f, 0.2765210271f),
                vec3(0.1335039884f, 0.7586020231f, 0.2184030265f),
                vec3(0.1949869841f, 0.8124759793f, 0.09765503556f),
                vec3(0.1855419874f, 0.8792020082f, 0.0110050384f),
                vec3(0.2303149849f, 0.8842020035f, -0.1274499595f),
                vec3(0.2604709566f, 0.6243780255f, 0.2550560236f),
                vec3(0.3515669703f, 0.6639760137f, -0.04219397157f),
                vec3(0.2537419796f, 0.4337159991f, 0.3235240281f),
                vec3(0.4116119742f, 0.4382640123f, 0.1322990209f),
                vec3(0.2705129683f, 0.289983958f, 0.3565300107f),
                vec3(0.4221459627f, 0.2731299996f, 0.162819013f),
                vec3(0.1647239923f, -0.208912015f, 0.2374899834f),
                vec3(0.2538059652f, -0.2406399995f, 0.09289999306f),
                vec3(0.2036079913f, -0.09622301161f, 0.2845970094f),
                vec3(0.2410059869f, 0.1713959873f, 0.3430930078f),
                vec3(0.3560759723f, 0.1434430033f, 0.1492879987f),
                vec3(0.3376559615f, -0.06637400389f, 0.1319919974f)
            };

            std::vector<int> bunny_indices =
            {
                126, 134, 133,
                342, 138, 134,
                133, 134, 138,
                126, 342, 134,
                312, 316, 317,
                169, 163, 162,
                312, 317, 319,
                312, 319, 318,
                169, 162, 164,
                169, 168, 163,
                312, 314, 315,
                169, 164, 165,
                169, 167, 168,
                312, 315, 316,
                312, 313, 314,
                169, 165, 166,
                169, 166, 167,
                312, 318, 313,
                308, 304, 305,
                308, 305, 306,
                179, 181, 188,
                177, 173, 175,
                177, 175, 176,
                302, 293, 300,
                322, 294, 304,
                188, 176, 175,
                188, 175, 179,
                158, 177, 187,
                305, 293, 302,
                305, 302, 306,
                322, 304, 308,
                188, 181, 183,
                158, 173, 177,
                293, 298, 300,
                304, 294, 296,
                304, 296, 305,
                185, 176, 188,
                185, 188, 183,
                187, 177, 176,
                187, 176, 185,
                305, 296, 298,
                305, 298, 293,
                436, 432,  28,
                436,  28,  23,
                434, 278, 431,
                30, 208, 209,
                30, 209,  29,
                19,  20,  24,
                208, 207, 211,
                208, 211, 209,
                19, 210, 212,
                433, 434, 431,
                433, 431, 432,
                433, 432, 436,
                436, 437, 433,
                277, 275, 276,
                277, 276, 278,
                209, 210,  25,
                21,  26,  24,
                21,  24,  20,
                25,  26,  27,
                25,  27,  29,
                435, 439, 277,
                439, 275, 277,
                432, 431,  30,
                432,  30,  28,
                433, 437, 438,
                433, 438, 435,
                434, 277, 278,
                24,  25, 210,
                24,  26,  25,
                29,  27,  28,
                29,  28,  30,
                19,  24, 210,
                208,  30, 431,
                208, 431, 278,
                435, 434, 433,
                435, 277, 434,
                25,  29, 209,
                27,  22,  23,
                27,  23,  28,
                26,  22,  27,
                26,  21,  22,
                212, 210, 209,
                212, 209, 211,
                207, 208, 278,
                207, 278, 276,
                439, 435, 438,
                12,   9,  10,
                12,  10,  13,
                2,   3,   5,
                2,   5,   4,
                16,  13,  14,
                16,  14,  17,
                22,  21,  16,
                13,  10,  11,
                13,  11,  14,
                1,   0,   3,
                1,   3,   2,
                15,  12,  16,
                19,  18,  15,
                19,  15,  16,
                19,  16,  20,
                9,   1,   2,
                9,   2,  10,
                3,   7,   8,
                3,   8,   5,
                16,  17,  23,
                16,  23,  22,
                21,  20,  16,
                10,   2,   4,
                10,   4,  11,
                0,   6,   7,
                0,   7,   3,
                12,  13,  16,
                451, 446, 445,
                451, 445, 450,
                442, 440, 439,
                442, 439, 438,
                442, 438, 441,
                421, 420, 422,
                412, 411, 426,
                412, 426, 425,
                408, 405, 407,
                413,  67,  68,
                413,  68, 414,
                391, 390, 412,
                80, 384, 386,
                404, 406, 378,
                390, 391, 377,
                390, 377,  88,
                400, 415, 375,
                398, 396, 395,
                398, 395, 371,
                398, 371, 370,
                112, 359, 358,
                112, 358, 113,
                351, 352, 369,
                125, 349, 348,
                345, 343, 342,
                342, 340, 339,
                341, 335, 337,
                328, 341, 327,
                331, 323, 333,
                331, 322, 323,
                327, 318, 319,
                327, 319, 328,
                315, 314, 324,
                302, 300, 301,
                302, 301, 303,
                320, 311, 292,
                285, 284, 289,
                310, 307, 288,
                310, 288, 290,
                321, 350, 281,
                321, 281, 282,
                423, 448, 367,
                272, 273, 384,
                272, 384, 274,
                264, 265, 382,
                264, 382, 383,
                440, 442, 261,
                440, 261, 263,
                252, 253, 254,
                252, 254, 251,
                262, 256, 249,
                262, 249, 248,
                228, 243, 242,
                228,  31, 243,
                213, 215, 238,
                213, 238, 237,
                19, 212, 230,
                224, 225, 233,
                224, 233, 231,
                217, 218,  56,
                217,  56,  54,
                217, 216, 239,
                217, 239, 238,
                217, 238, 215,
                218, 217, 215,
                218, 215, 214,
                6, 102, 206,
                186, 199, 200,
                197, 182, 180,
                170, 171, 157,
                201, 200, 189,
                170, 190, 191,
                170, 191, 192,
                175, 174, 178,
                175, 178, 179,
                168, 167, 155,
                122, 149, 158,
                122, 158, 159,
                135, 153, 154,
                135, 154, 118,
                143, 140, 141,
                143, 141, 144,
                132, 133, 136,
                130, 126, 133,
                124, 125, 127,
                122, 101, 100,
                122, 100, 121,
                110, 108, 107,
                110, 107, 109,
                98,  99,  97,
                98,  97,  64,
                98,  64,  66,
                87,  55,  57,
                83,  82,  79,
                83,  79,  84,
                78,  74,  50,
                49,  71,  41,
                49,  41,  37,
                49,  37,  36,
                58,  44,  60,
                60,  59,  58,
                51,  34,  33,
                39,  40,  42,
                39,  42,  38,
                243, 240,  33,
                243,  33, 229,
                39,  38,   6,
                44,  46,  40,
                55,  56,  57,
                64,  62,  65,
                64,  65,  66,
                41,  71,  45,
                75,  50,  51,
                81,  79,  82,
                77,  88,  73,
                93,  92,  94,
                68,  47,  46,
                96,  97,  99,
                96,  99,  95,
                110, 109, 111,
                111, 112, 110,
                114, 113, 123,
                114, 123, 124,
                132, 131, 129,
                133, 137, 136,
                135, 142, 145,
                145, 152, 135,
                149, 147, 157,
                157, 158, 149,
                164, 150, 151,
                153, 163, 168,
                153, 168, 154,
                185, 183, 182,
                185, 182, 184,
                161, 189, 190,
                200, 199, 191,
                200, 191, 190,
                180, 178, 195,
                180, 195, 196,
                102, 101, 204,
                102, 204, 206,
                43,  48, 104,
                43, 104, 103,
                216, 217,  54,
                216,  54,  32,
                207, 224, 231,
                230, 212, 211,
                230, 211, 231,
                227, 232, 241,
                227, 241, 242,
                235, 234, 241,
                235, 241, 244,
                430, 248, 247,
                272, 274, 253,
                272, 253, 252,
                439, 260, 275,
                225, 224, 259,
                225, 259, 257,
                269, 270, 407,
                269, 407, 405,
                270, 269, 273,
                270, 273, 272,
                273, 269, 268,
                273, 268, 267,
                273, 267, 266,
                273, 266, 265,
                273, 265, 264,
                448, 279, 367,
                281, 350, 368,
                285, 286, 301,
                290, 323, 310,
                290, 311, 323,
                282, 281, 189,
                292, 311, 290,
                292, 290, 291,
                307, 306, 302,
                307, 302, 303,
                316, 315, 324,
                316, 324, 329,
                331, 351, 350,
                330, 334, 335,
                330, 335, 328,
                341, 337, 338,
                344, 355, 354,
                346, 345, 348,
                346, 348, 347,
                364, 369, 352,
                364, 352, 353,
                365, 363, 361,
                365, 361, 362,
                376, 401, 402,
                373, 372, 397,
                373, 397, 400,
                376,  92, 377,
                381, 378, 387,
                381, 387, 385,
                386,  77,  80,
                390, 389, 412,
                416, 417, 401,
                403, 417, 415,
                408, 429, 430,
                419, 423, 418,
                427, 428, 444,
                427, 444, 446,
                437, 436, 441,
                450, 445,  11,
                450,  11,   4,
                447, 449,   5,
                447,   5,   8,
                441, 438, 437,
                425, 426, 451,
                425, 451, 452,
                417, 421, 415,
                408, 407, 429,
                399, 403, 400,
                399, 400, 397,
                394, 393, 416,
                389, 411, 412,
                386, 383, 385,
                408, 387, 378,
                408, 378, 406,
                377, 391, 376,
                94, 375, 415,
                372, 373, 374,
                372, 374, 370,
                359, 111, 360,
                359, 112, 111,
                113, 358, 349,
                113, 349, 123,
                346, 343, 345,
                343, 340, 342,
                338, 336, 144,
                338, 144, 141,
                327, 341, 354,
                327, 354, 326,
                331, 350, 321,
                331, 321, 322,
                314, 313, 326,
                314, 326, 325,
                300, 298, 299,
                300, 299, 301,
                288, 287, 289,
                189, 292, 282,
                287, 288, 303,
                284, 285, 297,
                368, 280, 281,
                448, 447, 279,
                274, 226, 255,
                267, 268, 404,
                267, 404, 379,
                429, 262, 430,
                439, 440, 260,
                257, 258, 249,
                257, 249, 246,
                430, 262, 248,
                234, 228, 242,
                234, 242, 241,
                237, 238, 239,
                237, 239, 236,
                15,  18, 227,
                15, 227, 229,
                222, 223,  82,
                222,  82,  83,
                214, 215, 213,
                214, 213,  81,
                38, 102,   6,
                122, 159, 200,
                122, 200, 201,
                174, 171, 192,
                174, 192, 194,
                197, 193, 198,
                190, 170, 161,
                181, 179, 178,
                181, 178, 180,
                166, 156, 155,
                163, 153, 152,
                163, 152, 162,
                120, 156, 149,
                120, 149, 121,
                152, 153, 135,
                140, 143, 142,
                135, 131, 132,
                135, 132, 136,
                130, 129, 128,
                130, 128, 127,
                100, 105, 119,
                100, 119, 120,
                106, 104, 107,
                106, 107, 108,
                91,  95,  59,
                93,  94,  68,
                91,  89,  92,
                76,  53,  55,
                76,  55,  87,
                81,  78,  79,
                74,  73,  49,
                69,  60,  45,
                58,  62,  64,
                58,  64,  61,
                53,  31,  32,
                32,  54,  53,
                42,  43,  38,
                35,  36,   0,
                35,   0,   1,
                34,  35,   1,
                34,   1,   9,
                44,  40,  41,
                44,  41,  45,
                33, 240,  51,
                63,  62,  58,
                63,  58,  59,
                45,  71,  70,
                76,  75,  51,
                76,  51,  52,
                86,  85,  84,
                86,  84,  87,
                89,  72,  73,
                89,  73,  88,
                91,  92,  96,
                91,  96,  95,
                72,  91,  60,
                72,  60,  69,
                104, 106, 105,
                119, 105, 117,
                119, 117, 118,
                124, 127, 128,
                117, 116, 129,
                117, 129, 131,
                118, 117, 131,
                135, 140, 142,
                146, 150, 152,
                146, 152, 145,
                149, 122, 121,
                166, 165, 151,
                166, 151, 156,
                158, 172, 173,
                161, 160, 189,
                199, 198, 193,
                199, 193, 191,
                204, 201, 202,
                178, 174, 194,
                200, 159, 186,
                109,  48,  67,
                48, 107, 104,
                216,  32, 236,
                216, 236, 239,
                223, 214,  81,
                223,  81,  82,
                33,  12,  15,
                32, 228, 234,
                32, 234, 236,
                240,  31,  52,
                256, 255, 246,
                256, 246, 249,
                258, 263, 248,
                258, 248, 249,
                275, 260, 259,
                275, 259, 276,
                207, 276, 259,
                270, 271, 429,
                270, 429, 407,
                413, 418, 366,
                413, 366, 365,
                368, 367, 279,
                368, 279, 280,
                303, 301, 286,
                303, 286, 287,
                283, 282, 292,
                283, 292, 291,
                320, 292, 189,
                298, 296, 297,
                298, 297, 299,
                318, 327, 326,
                318, 326, 313,
                329, 330, 317,
                336, 333, 320,
                326, 354, 353,
                334, 332, 333,
                334, 333, 336,
                342, 339, 139,
                342, 139, 138,
                345, 342, 126,
                347, 357, 356,
                369, 368, 351,
                363, 356, 357,
                363, 357, 361,
                366, 367, 368,
                366, 368, 369,
                375, 373, 400,
                92,  90, 377,
                409, 387, 408,
                386, 385, 387,
                386, 387, 388,
                412, 394, 391,
                396, 398, 399,
                408, 406, 405,
                415, 421, 419,
                415, 419, 414,
                425, 452, 448,
                425, 448, 424,
                444, 441, 443,
                448, 452, 449,
                448, 449, 447,
                446, 444, 443,
                446, 443, 445,
                250, 247, 261,
                250, 261, 428,
                421, 422, 423,
                421, 423, 419,
                427, 410, 250,
                417, 403, 401,
                403, 402, 401,
                420, 392, 412,
                420, 412, 425,
                420, 425, 424,
                386, 411, 389,
                383, 382, 381,
                383, 381, 385,
                378, 379, 404,
                372, 371, 395,
                372, 395, 397,
                371, 372, 370,
                361, 359, 360,
                361, 360, 362,
                368, 350, 351,
                349, 347, 348,
                356, 355, 344,
                356, 344, 346,
                344, 341, 340,
                344, 340, 343,
                338, 337, 336,
                328, 335, 341,
                324, 352, 351,
                324, 351, 331,
                320, 144, 336,
                314, 325, 324,
                322, 308, 309,
                310, 309, 307,
                287, 286, 289,
                203, 280, 279,
                203, 279, 205,
                297, 295, 283,
                297, 283, 284,
                447, 205, 279,
                274, 384,  80,
                274,  80, 226,
                266, 267, 379,
                266, 379, 380,
                225, 257, 246,
                225, 246, 245,
                256, 254, 253,
                256, 253, 255,
                430, 247, 250,
                226, 235, 244,
                226, 244, 245,
                232, 233, 244,
                232, 244, 241,
                230,  18,  19,
                32,  31, 228,
                219, 220,  86,
                219,  86,  57,
                226, 213, 235,
                206,   7,   6,
                122, 201, 101,
                201, 204, 101,
                180, 196, 197,
                170, 192, 171,
                200, 190, 189,
                194, 193, 195,
                183, 181, 180,
                183, 180, 182,
                155, 154, 168,
                149, 156, 151,
                149, 151, 148,
                155, 156, 120,
                145, 142, 143,
                145, 143, 146,
                136, 137, 140,
                133, 132, 130,
                128, 129, 116,
                100, 120, 121,
                110, 112, 113,
                110, 113, 114,
                66,  65,  63,
                66,  63,  99,
                66,  99,  98,
                96,  46,  61,
                89,  88,  90,
                86,  87,  57,
                80,  78,  81,
                72,  69,  49,
                67,  48,  47,
                67,  47,  68,
                56,  55,  53,
                50,  49,  36,
                50,  36,  35,
                40,  39,  41,
                242, 243, 229,
                242, 229, 227,
                6,  37,  39,
                42,  47,  48,
                42,  48,  43,
                61,  46,  44,
                45,  70,  69,
                69,  70,  71,
                69,  71,  49,
                74,  78,  77,
                83,  84,  85,
                73,  74,  77,
                93,  96,  92,
                68,  46,  93,
                95,  99,  63,
                95,  63,  59,
                115, 108, 110,
                115, 110, 114,
                125, 126, 127,
                129, 130, 132,
                137, 133, 138,
                137, 138, 139,
                148, 146, 143,
                148, 143, 147,
                119, 118, 154,
                161, 147, 143,
                165, 164, 151,
                158, 157, 171,
                158, 171, 172,
                159, 158, 187,
                159, 187, 186,
                194, 192, 191,
                194, 191, 193,
                189, 202, 201,
                182, 197, 184,
                205,   8,   7,
                48, 109, 107,
                218, 219,  57,
                218,  57,  56,
                207, 231, 211,
                232, 230, 231,
                232, 231, 233,
                53,  52,  31,
                388, 411, 386,
                409, 430, 250,
                262, 429, 254,
                262, 254, 256,
                442, 444, 428,
                273, 264, 383,
                273, 383, 384,
                429, 271, 251,
                429, 251, 254,
                413, 365, 362,
                67, 413, 360,
                282, 283, 295,
                285, 301, 299,
                202, 281, 280,
                284, 283, 291,
                284, 291, 289,
                320, 189, 160,
                308, 306, 307,
                307, 309, 308,
                319, 317, 330,
                319, 330, 328,
                353, 352, 324,
                332, 331, 333,
                340, 341, 338,
                354, 341, 344,
                349, 358, 357,
                349, 357, 347,
                364, 355, 356,
                364, 356, 363,
                364, 365, 366,
                364, 366, 369,
                374, 376, 402,
                375,  92, 373,
                77, 389, 390,
                382, 380, 381,
                389,  77, 386,
                393, 394, 412,
                393, 412, 392,
                401, 394, 416,
                415, 400, 403,
                411, 410, 427,
                411, 427, 426,
                422, 420, 424,
                247, 248, 263,
                247, 263, 261,
                445, 443,  14,
                445,  14,  11,
                449, 450,   4,
                449,   4,   5,
                443, 441,  17,
                443,  17,  14,
                436,  23,  17,
                436,  17, 441,
                424, 448, 422,
                448, 423, 422,
                414, 419, 418,
                414, 418, 413,
                406, 404, 405,
                399, 397, 395,
                399, 395, 396,
                420, 416, 392,
                388, 410, 411,
                386, 384, 383,
                390,  88,  77,
                375,  94,  92,
                415, 414,  68,
                415,  68,  94,
                370, 374, 402,
                370, 402, 398,
                361, 357, 358,
                361, 358, 359,
                125, 348, 126,
                346, 344, 343,
                340, 338, 339,
                337, 335, 334,
                337, 334, 336,
                325, 353, 324,
                324, 331, 332,
                324, 332, 329,
                323, 322, 309,
                323, 309, 310,
                294, 295, 297,
                294, 297, 296,
                289, 286, 285,
                202, 280, 203,
                288, 307, 303,
                282, 295, 321,
                67, 360, 111,
                418, 423, 367,
                418, 367, 366,
                272, 252, 251,
                272, 251, 271,
                272, 271, 270,
                255, 253, 274,
                265, 266, 380,
                265, 380, 382,
                442, 428, 261,
                440, 263, 258,
                440, 258, 260,
                409, 250, 410,
                255, 226, 245,
                255, 245, 246,
                31, 240, 243,
                236, 234, 235,
                236, 235, 237,
                233, 225, 245,
                233, 245, 244,
                220, 221,  85,
                220,  85,  86,
                81, 213, 226,
                81, 226,  80,
                7, 206, 205,
                186, 184, 198,
                186, 198, 199,
                204, 203, 205,
                204, 205, 206,
                195, 193, 196,
                171, 174, 172,
                173, 174, 175,
                173, 172, 174,
                155, 167, 166,
                160, 161, 143,
                160, 143, 144,
                119, 154, 155,
                148, 151, 150,
                148, 150, 146,
                140, 137, 139,
                140, 139, 141,
                127, 126, 130,
                114, 124, 128,
                114, 128, 115,
                117, 105, 106,
                117, 106, 116,
                104, 105, 100,
                104, 100, 103,
                59,  60,  91,
                97,  96,  61,
                97,  61,  64,
                91,  72,  89,
                87,  84,  79,
                87,  79,  76,
                78,  80,  77,
                49,  50,  74,
                60,  44,  45,
                61,  44,  58,
                51,  50,  35,
                51,  35,  34,
                39,  37,  41,
                33,  34,   9,
                33,   9,  12,
                0,  36,  37,
                0,  37,   6,
                40,  46,  47,
                40,  47,  42,
                53,  54,  56,
                65,  62,  63,
                72,  49,  73,
                79,  78,  75,
                79,  75,  76,
                52,  53,  76,
                92,  89,  90,
                96,  93,  46,
                102, 103, 100,
                102, 100, 101,
                116, 106, 108,
                116, 108, 115,
                123, 125, 124,
                116, 115, 128,
                118, 131, 135,
                140, 135, 136,
                148, 147, 149,
                120, 119, 155,
                164, 162, 152,
                164, 152, 150,
                157, 147, 161,
                157, 161, 170,
                186, 187, 185,
                186, 185, 184,
                193, 197, 196,
                202, 203, 204,
                194, 195, 178,
                198, 184, 197,
                67, 111, 109,
                38,  43, 103,
                38, 103, 102,
                214, 223, 222,
                214, 222, 221,
                214, 221, 220,
                214, 220, 219,
                214, 219, 218,
                213, 237, 235,
                221, 222,  83,
                221,  83,  85,
                15, 229,  33,
                227,  18, 230,
                227, 230, 232,
                52,  51, 240,
                75,  78,  50,
                408, 430, 409,
                260, 258, 257,
                260, 257, 259,
                224, 207, 259,
                268, 269, 405,
                268, 405, 404,
                413, 362, 360,
                447,   8, 205,
                299, 297, 285,
                189, 281, 202,
                290, 288, 289,
                290, 289, 291,
                322, 321, 295,
                322, 295, 294,
                333, 323, 311,
                333, 311, 320,
                317, 316, 329,
                320, 160, 144,
                353, 325, 326,
                329, 332, 334,
                329, 334, 330,
                339, 338, 141,
                339, 141, 139,
                348, 345, 126,
                347, 356, 346,
                123, 349, 125,
                364, 353, 354,
                364, 354, 355,
                365, 364, 363,
                376, 391, 394,
                376, 394, 401,
                92, 376, 374,
                92, 374, 373,
                377,  90,  88,
                380, 379, 378,
                380, 378, 381,
                388, 387, 409,
                388, 409, 410,
                416, 393, 392,
                399, 398, 402,
                399, 402, 403,
                250, 428, 427,
                421, 417, 416,
                421, 416, 420,
                426, 427, 446,
                426, 446, 451,
                444, 442, 441,
                452, 451, 450,
                452, 450, 449
            };
#endif

    } // namespace resource

} // namespace easy3d

